package gov.cms.grouper.snf.app;

import gov.cms.grouper.snf.SnfTables;
import gov.cms.grouper.snf.component.v100.SnfComponent;
import gov.cms.grouper.snf.lego.BlockWithException;
import gov.cms.grouper.snf.lego.SnfComparator;
import gov.cms.grouper.snf.lego.SnfUtils;
import gov.cms.grouper.snf.model.SnfError;
import gov.cms.grouper.snf.model.SnfProcessError;
import gov.cms.grouper.snf.model.reader.Rai300;
import gov.cms.grouper.snf.model.table.SnfVersionRow;
import gov.cms.grouper.snf.process.SnfValidations;
import gov.cms.grouper.snf.transfer.SnfClaim;
import gov.cms.grouper.snf.util.reader.SnfDataMapper;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;
import java.util.Scanner;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Pdpm {

  public static final Logger log = LoggerFactory.getLogger(Pdpm.class);

  private final SnfDataMapper mapper;
  private final List<String> errorLines;

  /*
   * Public API
   */
  public Pdpm() {
    this(new ArrayList<>());
  }

  /**
   * Process one MDS assessment through the PDPM grouper.
   *
   * @param line a fixed-length string in accordance with MDS 3.00.1 data specification
   * @return the SnfClaim object with grouping results or <code>null</code> if encountered issues
   *         Below are options to obtain the calculated results from the SnfClaim: - getHippsCode()
   *         : String - getErrors() : List<String> - getVersion() : int
   */
  public SnfClaim exec(String line) {
    SnfClaim claim = this.transformToClaim(line);
    return this.process(claim);
  }

  /**
   * Process one MDS assessment through the PDPM grouper.
   *
   * @param claim a SnfClaim object with the necessary inputs set for the PDPM calculation
   * @return the SnfClaim object with grouping results or <code>null</code> if encountered issues
   *         Below are options to obtain the calculated results from the SnfClaim: - getHippsCode()
   *         : String - getErrors() : List<String> - getVersion() : int
   */
  public SnfClaim exec(SnfClaim claim) {
    return this.process(claim);
  }

  /**
   * Process one or more MDS assessment through the PDPM grouper.
   *
   * @param snfFile a file containing one or more fixed-length strings representing MDS assessments
   * @return a list of SnfClaim objects with grouping results or <code>null</code> if encountered
   *         issues Below are options to obtain the calculated results from the SnfClaim: -
   *         getHippsCode() : String - getErrors() : List<String> - getVersion() : int
   */
  public List<SnfClaim> exec(File snfFile) {
    return this.processFile(snfFile, (line) -> {
      final SnfClaim claim = this.exec(line);
      return claim;
    });
  }

  public String getClaimFixedString(SnfClaim snfClaim) {
    return this.mapper.mapClaimToMds(snfClaim, Rai300.values());
  }

  /*
   * Protected constructor and methods primarily used for internal logic or testing
   */

  protected Pdpm(List<String> errorLines) {
    this.mapper = new SnfDataMapper();
    this.errorLines = errorLines;
  }

  protected SnfClaim process(SnfClaim claim) {
    tryExec(claim.getOriginalRecord(), () -> {
      Integer ver = this.getVersion(claim.getAssessmentReferenceDate());
      if (ver == null) {
        claim.addErrors(SnfError.INVALID_ASSESSMENT_REFERENCE_DATE.getReason());
      } else if (ver != 0) {
        try (SnfComponent snfComponent = new SnfComponent(ver)) {
          snfComponent.exec(claim);
        } catch (Throwable th) {
          throw th;
        }
      }
    });
    return claim;
  }

  protected <T> List<T> processFile(File file, Function<String, T> lineExec) {
    final List<T> results = new ArrayList<>(1000);

    try (Scanner sc = new Scanner(file, "UTF-8")) {
      // close the file on JVM shutdown
      Thread closeFileThread = new Thread() {
        public void run() {
          log.debug("closing files");
          SnfUtils.tryIgnoreExcption(() -> closeFile(sc));
        }
      };

      Runtime.getRuntime().addShutdownHook(closeFileThread);

      while (sc.hasNextLine()) {
        String line = sc.nextLine();
        if (sc.ioException() != null) {
          throw sc.ioException();
        }
        T item = lineExec.apply(line);
        results.add(item);
      }

    } catch (Throwable th) {
      throw new RuntimeException(th);
    }
    return results;

  }

  private void closeFile(Scanner sc) {
    sc.close();
  }

  protected Integer getVersion(LocalDate ard) {
    Integer version = null;

    for (Entry<Integer, SnfVersionRow> row : SnfTables.versions.entrySet()) {
      if (SnfComparator.betweenInclusive(row.getValue().getFrom(), ard, row.getValue().getTo())) {
        version = row.getKey();
        break;
      }
    }

    return version;
  }

  protected SnfClaim transformToClaim(String line) {
    SnfClaim claim = this.mapper.mapMdsToClaim(line, Rai300.values());
    return claim;
  }

  protected void tryExec(String line, BlockWithException ex) {
    try {
      ex.exec();
    } catch (Throwable th) {
      if (errorLines == null) {
        throw new RuntimeException(th);
      } else {
        errorLines.add(line);
      }
    }

  }

  protected SnfClaim process(Integer version, SnfClaim claim) {
    return this.process(version, claim, null);
  }

  protected SnfClaim process(Integer version, SnfClaim claim,
      Consumer<SnfClaim> additionalPostProcessing) {
    tryExec(claim.getOriginalRecord(), () -> {
      Integer ver = version;
      if (version == null) {
        ver = this.getVersion(claim.getAssessmentReferenceDate());
      }
      if (ver == null) {
        claim.addErrors(SnfError.INVALID_ASSESSMENT_REFERENCE_DATE.getReason());
      } else if (ver != 0) {
        try (SnfComponent snfComponent = new SnfComponent(ver)) {
          snfComponent.exec(claim);
        } catch (SnfProcessError se) {
          throw se;
        } catch (Throwable th) {
          claim.addErrors(SnfError.PROCESS_ERROR.getReason(th.getMessage()));
          throw th;
        }
      }
    });

    if (additionalPostProcessing != null) {
      additionalPostProcessing.accept(claim);
    }
    return claim;
  }

  protected SnfClaim exec(Integer version, String line,
      Consumer<SnfClaim> additionalPostProcessing) {
    SnfClaim claim = this.transformToClaim(line);
    this.execClaim(version, claim, additionalPostProcessing);
    return claim;
  }

  protected List<SnfClaim> exec(Integer version, List<String> lines,
      Consumer<SnfClaim> additionalPostProcessing) {
    final List<SnfClaim> results = new ArrayList<>(lines.size());

    for (String line : lines) {
      final SnfClaim claim = this.exec(version, line, additionalPostProcessing);
      results.add(claim);
    }

    return results;

  }

  protected List<SnfClaim> exec(Integer version, List<String> lines) {
    return this.exec(version, lines, null);
  }


  protected SnfClaim exec(Integer version, String line) {
    return this.exec(version, line, null);
  }

  protected List<SnfClaim> exec(Integer version, Path path) throws IOException {
    final List<SnfClaim> results = this.processFile(version, path, (line) -> {
      final SnfClaim claim = this.exec(version, line, null);
      return claim;
    });

    return results;
  }

  protected List<SnfClaim> exec(Integer version, Path path,
      Consumer<SnfClaim> additionalPostProcessing) {
    List<SnfClaim> claims = this.processFile(version, path, (line) -> {
      final SnfClaim claim = this.exec(version, line, additionalPostProcessing);
      return claim;
    });

    return claims;

  }

  protected SnfClaim execClaim(Integer version, SnfClaim claim) {
    return this.execClaim(version, claim, null);
  }

  protected SnfClaim execClaim(Integer version, SnfClaim claim,
      Consumer<SnfClaim> additionalPostProcessing) {
    this.validate(claim);
    this.process(version, claim, additionalPostProcessing);
    return claim;
  }


  protected <T> List<T> processFile(Integer version, Path path, Function<String, T> lineExec) {
    final List<T> results = new ArrayList<>(1000);

    try (Scanner sc = new Scanner(path, "UTF-8")) {
      // close the file on JVM shutdown
      Thread closeFileThread = new Thread() {
        public void run() {
          log.debug("closing files");
          SnfUtils.tryIgnoreExcption(() -> closeFile(sc));
        }
      };

      Runtime.getRuntime().addShutdownHook(closeFileThread);

      int counter = 1;

      while (sc.hasNextLine()) {
        if (counter % 10000 == 0) {
          System.out.println("lines processed: " + counter);
        }
        String line = sc.nextLine();
        if (sc.ioException() != null) {
          throw sc.ioException();
        }
        T item = lineExec.apply(line);
        results.add(item);
        counter += 1;
      }

    } catch (Throwable th) {
      throw new RuntimeException(th);
    }
    return results;

  }

  protected void validate(SnfClaim claim) {
    tryExec(claim.getOriginalRecord(), () -> {
      SnfValidations.validateInputs(claim);
    });
  }


}
